using UnityEditor;
using UnityEngine;
using System.IO;
using System;
using System.Collections.Generic;
using System.Xml;
using Unity.SharpZipLib.Zip;
using UnityEditor.SceneManagement;
using Random = UnityEngine.Random;

[System.Serializable]
public class MapsWrapper
{
    public MapConfiguration[] maps;
}

[System.Serializable]
public class MapConfiguration
{
    public string MapTitleEn;
    public string MapTitleRu;
    public string DescriptionEn;
    public string DescriptionRu;
    public string Author;
    public string ScreenshotPath;
    public bool IsForPowerfulPhone;
    public string MinimalGameVersion;
    public string MapVersion;
    public string ScenePath;
    public string ResourceFolderPath;
}

[System.Serializable]
public class DataSettings
{
    public bool SetLayout;
}

public class MapToolsApi
{
    static DataSettings _dataSettings;
    public static List<MapConfiguration> Maps = new List<MapConfiguration>();
    public static int SelectedMapIndex = 0;
    static string mapsDataFilePath = Path.Combine(Application.dataPath, "MapTools/Editor/Resources/mapsData.json");
    static string DataFilePath = Path.Combine(Application.dataPath, "MapTools/Editor/Resources/Data.json");
    static string ScreenshotFileName = string.Empty;

    [InitializeOnLoad]
    static class StartupPlatformSwitcher
    {
        static StartupPlatformSwitcher()
        {
            EditorApplication.delayCall += SwitchPlatform;
            EditorApplication.delayCall += SetLayout;
        }

        private static void SwitchPlatform()
        {
            if (!SessionState.GetBool("FirstInitDone", false))
            {
                SessionState.SetBool("FirstInitDone", true);
                if (EditorUserBuildSettings.activeBuildTarget != BuildTarget.Android &&
                    EditorUserBuildSettings.activeBuildTarget != BuildTarget.iOS)
                {
                    Debug.Log("Automatically switching platform from Windows to Android...");
                    if (EditorUserBuildSettings.SwitchActiveBuildTarget(BuildTargetGroup.Android, BuildTarget.Android))
                    {
                        Debug.Log("Platform switched to Android successfully.");
                    }
                    else
                    {
                        Debug.LogError("Failed to switch to Android. Please install Android support via File -> Build Settings -> Platform.");
                    }
                }
            }
        }
    }

    private static DataSettings LoadDataSettings()
    {
        if (_dataSettings != null) return _dataSettings;
        if (File.Exists(DataFilePath))
        {
            string json = File.ReadAllText(DataFilePath);
            if (string.IsNullOrEmpty(json))
            {
                _dataSettings = new DataSettings();
            }
            else
            {
                _dataSettings = JsonUtility.FromJson<DataSettings>(json);
            }
        }

        if (_dataSettings == null)
            _dataSettings = new DataSettings();
        return _dataSettings;
    }

    static void ClearAssetBundle(string bundleName)
    {
        string[] names = AssetDatabase.GetAllAssetBundleNames();

        foreach (string name in names)
        {
            if (name == bundleName)
                AssetDatabase.RemoveAssetBundleName(bundleName, true);
        }

        string[] assetPaths = AssetDatabase.GetAssetPathsFromAssetBundle(bundleName);
        foreach (string assetPath in assetPaths)
        {
            AssetImporter importer = AssetImporter.GetAtPath(assetPath);
            if (importer != null)
            {
                importer.assetBundleName = "";
                EditorUtility.SetDirty(importer);
            }
        }
    }

    static void UpdateCurrentScene()
    {
        EditorSceneManager.SaveOpenScenes();
        AssetDatabase.StartAssetEditing();
        ClearAssetBundle("scene");
        MapConfiguration config = GetCurrentMap();
        SceneAsset sceneAsset = !string.IsNullOrEmpty(config.ScenePath) ? AssetDatabase.LoadAssetAtPath<SceneAsset>(config.ScenePath) : null;
        if (sceneAsset != null)
        {
            string newPath = AssetDatabase.GetAssetPath(sceneAsset);
            config.ScenePath = newPath;
            AssetImporter newImporter = AssetImporter.GetAtPath(newPath);
            if (newImporter != null)
            {
                newImporter.assetBundleName = "scene";
                EditorUtility.SetDirty(newImporter);
                Debug.Log("AssetBundle 'scene' assigned for " + newPath);
            }
        }
        else config.ScenePath = "";

        AssetDatabase.StopAssetEditing();
    }

    static void UpdateCurrentFolder()
    {
        AssetDatabase.StartAssetEditing();
        ClearAssetBundle("resource");
        MapConfiguration config = MapToolsApi.GetCurrentMap();
        DefaultAsset resourceFolder = !string.IsNullOrEmpty(config.ResourceFolderPath) ? AssetDatabase.LoadAssetAtPath<DefaultAsset>(config.ResourceFolderPath) : null;
        if (resourceFolder != null)
        {
            string newFolderPath = AssetDatabase.GetAssetPath(resourceFolder);
            AssignAssetBundleToFolder(newFolderPath, "resource");
        }
        else config.ResourceFolderPath = "";

        AssetDatabase.StopAssetEditing();
    }

    static void AssignAssetBundleToFolder(string folderPath, string bundleName)
    {
        string[] guids = AssetDatabase.FindAssets("", new[] { folderPath });
        foreach (string guid in guids)
        {
            string assetPath = AssetDatabase.GUIDToAssetPath(guid);
            if (AssetDatabase.IsValidFolder(assetPath))
                continue;
            AssetImporter importer = AssetImporter.GetAtPath(assetPath);
            if (importer != null)
            {
                importer.assetBundleName = bundleName;
                EditorUtility.SetDirty(importer);
            }
        }

        Debug.Log("AssetBundle '" + bundleName + "' assigned to all files in the folder " + folderPath);
    }

    private static void SaveDataSettings()
    {
        if (_dataSettings == null) return;
        string json = JsonUtility.ToJson(_dataSettings, true);
        string directory = Path.GetDirectoryName(DataFilePath);
        if (!Directory.Exists(directory))
            Directory.CreateDirectory(directory);
        File.WriteAllText(DataFilePath, json);
    }

    private static void SetLayout()
    {
        DataSettings settings = LoadDataSettings();
        if (!settings.SetLayout)
        {
            MapToolsWLTLayout.LoadLayout("MapToolsLayout");
            settings.SetLayout = true;
            SaveDataSettings();
        }
    }

    public static void BuildAllAssetBundlesIos()
    {
        if (!BuildPipeline.IsBuildTargetSupported(BuildTargetGroup.iOS, BuildTarget.iOS))
        {
            Debug.LogError("iOS platform support is not installed. Install it through the Unity Hub.");
            return;
        }

        MapConfiguration map = GetCurrentMap();
        Debug.Log("Building iOS AssetBundles for map '" + (map != null ? map.MapTitleEn : "") + "'");
        if (EditorUserBuildSettings.SwitchActiveBuildTarget(BuildTargetGroup.iOS, BuildTarget.iOS))
            BuildAllAssetBundles();
        else
            Debug.LogError("Unable to switch platform to iOS.");
    }

    public static void BuildAllAssetBundlesAndroid()
    {
        if (!BuildPipeline.IsBuildTargetSupported(BuildTargetGroup.Android, BuildTarget.Android))
        {
            Debug.LogError("Android platform support is not installed. Install it through the Unity Hub.");
            return;
        }

        MapConfiguration map = GetCurrentMap();
        Debug.Log("Building Android AssetBundles for map '" + (map != null ? map.MapTitleEn : "") + "'");
        if (EditorUserBuildSettings.SwitchActiveBuildTarget(BuildTargetGroup.Android, BuildTarget.Android))
            BuildAllAssetBundles();
        else
            Debug.LogError("Unable to switch platform to Android.");
    }

    static void BuildAllAssetBundles()
    {
        UpdateCurrentFolder();
        UpdateCurrentScene();

        if (!AreAssetBundleNamesValid())
            return;

        MapConfiguration config = GetCurrentMap();
        string tempDir = Path.Combine("Temp", EditorUserBuildSettings.activeBuildTarget.ToString());
        string modDir = "MapsBuilds";

        if (!Directory.Exists(tempDir))
            Directory.CreateDirectory(tempDir);

        if (!Directory.Exists(modDir))
            Directory.CreateDirectory(modDir);

        AssetBundleManifest manifest = BuildPipeline.BuildAssetBundles(tempDir, BuildAssetBundleOptions.None, EditorUserBuildSettings.activeBuildTarget);
        bool isSuccess = true;

        if (manifest == null)
        {
            isSuccess = false;
        }

        string[] expectedFiles = { "resource", "scene" };
        foreach (string fileName in expectedFiles)
        {
            string filePath = Path.Combine(tempDir, fileName);
            if (!File.Exists(filePath))
            {
                isSuccess = false;
            }
        }

        if (!isSuccess)
        {
            Debug.LogError($"AssetBundles build for {EditorUserBuildSettings.activeBuildTarget}. Failed.");
            return;
        }
        else
        {
            Debug.Log($"AssetBundles build for {EditorUserBuildSettings.activeBuildTarget}. Converting to zip...");
        }

        try
        {
            ConvertToMod(tempDir, modDir);
            Debug.Log("Successfully converted AssetBundles to zip.");
        }
        catch (Exception ex)
        {
            Debug.LogError($"Failed to convert AssetBundles to zip: {ex.Message}");
        }
    }


    static bool AreAssetBundleNamesValid()
    {
        string[] names = AssetDatabase.GetAllAssetBundleNames();
        bool hasResource = false;
        bool hasScene = false;

        foreach (string name in names)
        {
            if (name.Equals("resource", StringComparison.OrdinalIgnoreCase))
                hasResource = true;
            if (name.Equals("scene", StringComparison.OrdinalIgnoreCase))
                hasScene = true;
        }

        if (!hasResource)
        {
            Debug.LogWarning("AssetBundle 'resource' is missing. Please fix it.");
        }

        if (!hasScene)
        {
            Debug.LogWarning("AssetBundle 'scene' is missing. Please fix it.");
        }

        return hasResource && hasScene;
    }


    static void ConvertToMod(string tempPath, string modPath)
    {
        bool isAndroidBuild = EditorUserBuildSettings.activeBuildTarget == BuildTarget.Android;
        bool isIosBuild = EditorUserBuildSettings.activeBuildTarget == BuildTarget.iOS;
        if (!isAndroidBuild && !isIosBuild)
        {
            Debug.LogError("Unknown activeBuildTarget");
            return;
        }

        try
        {
            AddXmlIfMissing(tempPath);
            XmlInject(tempPath, isAndroidBuild, isIosBuild);
            CopyScreenshot(tempPath);
            CreateZip(tempPath, modPath, isAndroidBuild, isIosBuild);
        }
        finally
        {
            DeleteDirectory(tempPath);
        }
    }

    static void DeleteDirectory(string path)
    {
        ScreenshotFileName = string.Empty;
        if (Directory.Exists(path))
            Directory.Delete(path, true);
    }


    static void AddXmlIfMissing(string modPath)
    {
        DirectoryInfo dir = new DirectoryInfo(modPath);
        bool xmlExists = false;
        foreach (FileInfo file in dir.GetFiles())
        {
            if (file.Name.Equals("map.xml", StringComparison.OrdinalIgnoreCase))
            {
                xmlExists = true;
                break;
            }
        }

        if (!xmlExists)
        {
            TextAsset xmlAsset = Resources.Load<TextAsset>("map");
            if (xmlAsset != null)
            {
                File.WriteAllBytes(Path.Combine(modPath, "map.xml"), xmlAsset.bytes);
            }
            else
            {
                Debug.LogWarning("map.xml resource not found in Resources folder.");
            }
        }
    }

    static void UpdateXmlNode(XmlDocument xmlDoc, string nodeName, string newValue)
    {
        XmlNode node = xmlDoc.SelectSingleNode($"/MapsSettings/{nodeName}");
        if (node != null)
        {
            node.InnerText = newValue;
        }
        else
        {
            Debug.LogWarning($"Node {nodeName} not found in XML.");
        }
    }

    static void XmlInject(string modPath, bool isAndroid, bool isIos)
    {
        MapConfiguration config = MapToolsApi.GetCurrentMap();
        string xmlPath = Path.Combine(modPath, "map.xml");

        string nameMapRu = config.MapTitleRu;
        string nameMapEn = config.MapTitleEn;
        string descriptionRu = config.DescriptionRu;
        string descriptionEn = config.DescriptionEn;
        string author = config.Author;
        string minimalVersionGame = config.MinimalGameVersion;
        string mapVersion = config.MapVersion;
        int mapForPowerfulPhone = config.IsForPowerfulPhone ? 1 : 0;

        if (string.IsNullOrEmpty(nameMapRu) ||
            string.IsNullOrEmpty(nameMapEn) ||
            string.IsNullOrEmpty(descriptionRu) ||
            string.IsNullOrEmpty(descriptionEn) ||
            string.IsNullOrEmpty(author) ||
            string.IsNullOrEmpty(minimalVersionGame) ||
            string.IsNullOrEmpty(mapVersion))
        {
            Debug.LogError("Some map settings are not set or are empty. Use sample map.xml");
            return;
        }

        if (!File.Exists(xmlPath))
        {
            Debug.LogWarning("map.xml file not found in mod folder.");
            return;
        }

        XmlDocument xmlDoc = new XmlDocument();
        try
        {
            xmlDoc.Load(xmlPath);
        }
        catch (Exception ex)
        {
            Debug.LogError($"Error loading XML file by path {xmlPath}: {ex.Message}");
            return;
        }

        UpdateXmlNode(xmlDoc, "NameMapRu", nameMapRu);
        UpdateXmlNode(xmlDoc, "NameMapEn", nameMapEn);
        UpdateXmlNode(xmlDoc, "DescriptionRu", descriptionRu);
        UpdateXmlNode(xmlDoc, "DescriptionEn", descriptionEn);
        UpdateXmlNode(xmlDoc, "Author", author);
        UpdateXmlNode(xmlDoc, "MapForPowerfulPhone", mapForPowerfulPhone.ToString());
        UpdateXmlNode(xmlDoc, "MinimalVersionGame", minimalVersionGame);
        UpdateXmlNode(xmlDoc, "MapVersion", mapVersion);
        UpdateXmlNode(xmlDoc, "IsSupportIos", isAndroid ? "0" : isIos ? "1" : "0");

        try
        {
            xmlDoc.Save(xmlPath);
        }
        catch (Exception ex)
        {
            Debug.LogError($"Error saving XML file: {ex.Message}");
            return;
        }
    }

    static void CopyScreenshot(string modPath)
    {
        string xmlPath = Path.Combine(modPath, "map.xml");
        MapConfiguration config = MapToolsApi.GetCurrentMap();
        string screenshotPath = config.ScreenshotPath;
        if (string.IsNullOrEmpty(screenshotPath) || !File.Exists(screenshotPath))
        {
            Debug.LogError("Screenshot path is invalid or file not found: " + screenshotPath);
            return;
        }

        string fileName = Path.GetFileName(screenshotPath);
        string extension = Path.GetExtension(screenshotPath);
        string newFileName = "Screenshot" + extension;
        string destPath = Path.Combine(modPath, newFileName);

        try
        {
            File.Copy(screenshotPath, destPath, true);
        }
        catch (Exception ex)
        {
            Debug.LogError("Error copying screenshot: " + ex.Message);
            return;
        }

        try
        {
            XmlDocument xmlDoc = new XmlDocument();
            xmlDoc.Load(xmlPath);
            UpdateXmlNode(xmlDoc, "FileScreenshotName", newFileName);
            xmlDoc.Save(xmlPath);
        }
        catch (Exception ex)
        {
            Debug.LogError("Error updating XML file: " + ex.Message);
            return;
        }

        ScreenshotFileName = newFileName;
    }

    static void CreateZip(string sourceFolder, string outputFolder, bool isAndroidBuild, bool isIosBuild)
    {
        MapConfiguration config = MapToolsApi.GetCurrentMap();
        string mapNameNoSpaces = config.MapTitleEn.Replace(" ", "");
        string zipFileName = isAndroidBuild
            ? "Android_Map(" + mapNameNoSpaces + ").zip"
            : "iOS_Map(" + mapNameNoSpaces + ").zip";
        string zipFilePath = Path.Combine(outputFolder, zipFileName);

        try
        {
            using (FileStream fs = File.Create(zipFilePath))
            using (ZipOutputStream zipStream = new ZipOutputStream(fs))
            {
                zipStream.SetLevel(9);
                foreach (string file in Directory.GetFiles(sourceFolder, "*", SearchOption.AllDirectories))
                {
                    string fileName = Path.GetFileName(file);
                    bool isResource = fileName.Equals("resource", StringComparison.OrdinalIgnoreCase);
                    bool isScene = fileName.Equals("scene", StringComparison.OrdinalIgnoreCase);
                    bool isXml = fileName.Equals("map.xml", StringComparison.OrdinalIgnoreCase);
                    bool isScreenshot = !string.IsNullOrEmpty(ScreenshotFileName) &&
                                        fileName.Equals(ScreenshotFileName, StringComparison.OrdinalIgnoreCase);

                    if (isResource)
                    {
                        fileName = "Resource";
                    }
                    else if (isScene)
                    {
                        fileName = "Scene";
                    }

                    if (isResource || isScene || isScreenshot || isXml)
                    {
                        ZipEntry newEntry = new ZipEntry(fileName)
                        {
                            DateTime = DateTime.Now,
                            Size = new FileInfo(file).Length
                        };

                        zipStream.PutNextEntry(newEntry);
                        using (FileStream fileStream = File.OpenRead(file))
                        {
                            fileStream.CopyTo(zipStream);
                        }

                        zipStream.CloseEntry();
                    }
                }

                zipStream.Finish();
            }

            Debug.Log($"The files are packaged in a zip archive: {zipFilePath}");
            EditorUtility.RevealInFinder(zipFilePath);
        }
        catch (Exception ex)
        {
            Debug.LogError($"Error when creating a zip archive: {ex.Message}");
        }
    }

    public static void LoadMaps()
    {
        if (File.Exists(mapsDataFilePath))
        {
            string json = File.ReadAllText(mapsDataFilePath);
            MapsWrapper wrapper = JsonUtility.FromJson<MapsWrapper>(json);
            if (wrapper != null && wrapper.maps != null)
                Maps = new List<MapConfiguration>(wrapper.maps);
        }

        if (Maps == null)
            Maps = new List<MapConfiguration>();
    }

    public static void SaveMaps()
    {
        HashSet<string> uniqueNames = new HashSet<string>(StringComparer.OrdinalIgnoreCase);

        foreach (var map in Maps)
        {
            if (!uniqueNames.Add(map.MapTitleEn))
            {
                Debug.LogWarning($"A map {map.MapTitleEn} is duplicate.");
                map.MapTitleEn = "Duplicate" + Random.Range(1000, 99999);
            }
        }

        MapsWrapper wrapper = new MapsWrapper();
        wrapper.maps = Maps.ToArray();
        string json = JsonUtility.ToJson(wrapper, true);
        string directory = Path.GetDirectoryName(mapsDataFilePath);
        if (!Directory.Exists(directory))
            Directory.CreateDirectory(directory);
        File.WriteAllText(mapsDataFilePath, json);
    }

    public static MapConfiguration GetCurrentMap()
    {
        if (Maps != null && Maps.Count > 0 && SelectedMapIndex >= 0 && SelectedMapIndex < Maps.Count)
            return Maps[SelectedMapIndex];
        return null;
    }

    public static void AddMap()
    {
        foreach (MapConfiguration map in Maps)
        {
            if (map.MapTitleEn == "New Map")
            {
                Debug.LogWarning("Map with name 'New Map' already exists.");
                SelectedMapIndex = Maps.IndexOf(map);
                return;
            }
        }

        MapConfiguration newMap = new MapConfiguration();
        newMap.MapTitleEn = "New Map";
        newMap.MapTitleRu = "Новая карта";
        newMap.DescriptionEn = "Here is the description in English";
        newMap.DescriptionRu = "Здесь описание на Русском языке";
        newMap.Author = "Unknown";
        newMap.ScreenshotPath = "";
        newMap.IsForPowerfulPhone = true;
        newMap.MinimalGameVersion = "1.3.1";
        newMap.MapVersion = "0.1";
        newMap.ScenePath = "";
        newMap.ResourceFolderPath = "";
        Maps.Add(newMap);
        SelectedMapIndex = Maps.Count - 1;
        SaveMaps();
    }
}